import json
import logging
import abc
from config import LAYOUT_FILES
from cse_image_items.Descriptor import Descriptor
from cse_image_items.CSERegion import CSERegion
from cse_image_items.MetaData import MetaData


logger = logging.getLogger(__name__)

def load_bin_image_to_bytes(image):
    """Loads a binary image as bytes

    Returns:
        bytes -- CSE FW image bytes
    """
    logger.info('START load_bin_image_to_bytes FUNCTION')
    logger.debug('Image path is [%s]' %(image))

    try: 
        with open(image, 'rb') as image_file:
            return image_file.read()
    except IOError:
        logger.exception('Image file does not exist')
        raise IOError('Image file does not exist')


def load_layout_file(layout):
    """Load the image layout as json

     Returns:
        json -- the image layout
    """ 
    logger.info('STARTED load_layout_file')
    logger.debug('Layout file is [%s]' %(layout))

    try:
        layout_file= LAYOUT_FILES[layout]
        with open('layout/' + layout_file) as layout_json:
            try:
                return json.load(layout_json)
            except TypeError:
                logger.exception('Layout file is invalid (JSON formmat)')
                raise TypeError('Layout file is invalid (JSON formmat)')
    except IOError:
        logger.exception('Layout file does not exist')
        raise IOError('Layout file does not exist')


class ImageFactory(object):
    @staticmethod
    def create(image_path, layout_path, CSE_region_only):
        logger.info('START ImageFactory.create FUNCTION')

        data = load_bin_image_to_bytes(image_path)
        layout = load_layout_file(layout_path)
        if CSE_region_only:
            logger.debug('Creating layout for image type: CSE region image')
            return CSERegionImage(data, layout)
        else:
            logger.debug('Creating layout for image type: full image')
            return Image(data, layout)
    

ABC = abc.ABCMeta('ABC', (object,), {})
class AbstractImage(ABC):
    @abc.abstractmethod
    def __init__(self, data, layout):
        logger.info('START AbstractImage.__init__ FUNCTION')

        self.cse_region = CSERegion(data, **layout['cse_region'])
        self.metadata = MetaData(self.cse_region)



class Image(AbstractImage):
    def __init__(self, data, layout):
        logger.info('START Image.__init__ FUNCTION')

        self.descriptor = Descriptor(data, **layout['descriptor'])
        cse_region_offset = self.descriptor.cse_offset.start
        cse_region_size = hex(
            self.descriptor.cse_offset.end.int_value() -
            cse_region_offset.int_value()
        )
        layout['cse_region']['offset'] = cse_region_offset.hex_value()
        layout['cse_region']['size'] = cse_region_size
        super(Image, self).__init__(data, layout)


class CSERegionImage(AbstractImage):
    def __init__(self, data, layout):
        logger.info('START CSERegionImage.__init__ FUNCTION')

        layout['cse_region']['offset'] = u'0x0'
        layout['cse_region']['size'] = hex(len(data)).decode()
        super(Image, self).__init__(data, layout)
